import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Camera, User, Calendar, Star } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MySessions.css'

const MySessions = () => {
  const [sessions, setSessions] = useState([])

  useEffect(() => {
    loadSessions()
    const interval = setInterval(loadSessions, 500)
    return () => clearInterval(interval)
  }, [])

  const loadSessions = () => {
    const saved = localStorage.getItem('photoSessions')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setSessions(parsed)
      } catch (e) {
        setSessions([])
      }
    } else {
      const defaultSessions = [
        {
          id: 1,
          photographer: 'Анна Петрова',
          theme: 'Love Story',
          date: '2025-03-15',
          rating: 5
        },
        {
          id: 2,
          photographer: 'Михаил Соколов',
          theme: 'Портрет',
          date: '2025-03-10',
          rating: 4
        },
        {
          id: 3,
          photographer: 'Елена Волкова',
          theme: 'Семейная',
          date: '2025-03-05',
          rating: 5
        }
      ]
      setSessions(defaultSessions)
      localStorage.setItem('photoSessions', JSON.stringify(defaultSessions))
    }
  }

  const renderStars = (rating) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        size={16}
        fill={i < rating ? '#A78BFA' : 'none'}
        stroke={i < rating ? '#A78BFA' : 'var(--text-light)'}
        strokeWidth={1.5}
      />
    ))
  }

  const averageRating = sessions.length > 0 
    ? (sessions.reduce((sum, s) => sum + s.rating, 0) / sessions.length).toFixed(1)
    : 0

  const totalSessions = sessions.length
  const uniquePhotographers = new Set(sessions.map(s => s.photographer)).size

  return (
    <div className="my-sessions-page">
      <div className="container">
        <motion.div 
          className="page-header"
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <h1 className="page-title">Мои съёмки</h1>
          <p className="page-subtitle">История ваших фотосессий</p>
        </motion.div>

        <motion.div 
          className="stats-minimal"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.5, delay: 0.1 }}
        >
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{totalSessions}</div>
            <div className="stat-minimal-label">Съёмок</div>
          </div>
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{uniquePhotographers}</div>
            <div className="stat-minimal-label">Фотографов</div>
          </div>
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{averageRating}</div>
            <div className="stat-minimal-label">Рейтинг</div>
          </div>
        </motion.div>

        <ImageSlider images={[
          '/images/photo-1.jpg',
          '/images/photo-2.jpg',
          '/images/photo-3.jpg',
          '/images/photo-4.jpg'
        ]} />

        <motion.section 
          className="sessions-list"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          {sessions.length > 0 ? (
            <div className="sessions-vertical">
              {sessions.map((session, index) => (
                <motion.div 
                  key={session.id} 
                  className="session-item"
                  initial={{ opacity: 0, x: -30 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.4 }}
                  whileHover={{ x: 8, borderColor: 'var(--primary)' }}
                >
                  <div className="session-item-left">
                    <div className="session-item-icon">
                      <Camera size={24} strokeWidth={2} />
                    </div>
                    <div className="session-item-content">
                      <div className="session-item-header">
                        <h3 className="session-item-theme">{session.theme}</h3>
                        <div className="session-item-rating">
                          {renderStars(session.rating)}
                        </div>
                      </div>
                      <div className="session-item-details">
                        <div className="session-item-detail">
                          <User size={16} strokeWidth={2} />
                          <span>{session.photographer}</span>
                        </div>
                        <div className="session-item-detail">
                          <Calendar size={16} strokeWidth={2} />
                          <span>
                            {new Date(session.date).toLocaleDateString('ru-RU', { 
                              day: 'numeric', 
                              month: 'long', 
                              year: 'numeric' 
                            })}
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <Camera size={56} strokeWidth={1.5} className="empty-icon" />
              <h3 className="empty-title">Нет запланированных съёмок</h3>
              <p className="empty-text">Запишитесь на фотосессию</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="process-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <h2 className="section-title">Процесс работы</h2>
          <div className="process-steps">
            <div className="process-step">
              <div className="process-step-number">01</div>
              <h3 className="process-step-title">Запись</h3>
              <p className="process-step-text">Выберите стиль и дату</p>
            </div>
            <div className="process-step">
              <div className="process-step-number">02</div>
              <h3 className="process-step-title">Подготовка</h3>
              <p className="process-step-text">Обсуждение деталей</p>
            </div>
            <div className="process-step">
              <div className="process-step-number">03</div>
              <h3 className="process-step-title">Съёмка</h3>
              <p className="process-step-text">Профессиональная работа</p>
            </div>
            <div className="process-step">
              <div className="process-step-number">04</div>
              <h3 className="process-step-title">Результат</h3>
              <p className="process-step-text">Готовые фото</p>
            </div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MySessions

